//
// Primary class for PTC 2.0 C++ API
// Copyright (c) 1998 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
//

// include files
#include "Copy/Copy.h"
#include "Core/Error.h"
#include "DirectX/Check.h"
#include "DirectX/Primary.h"

// using directive
using namespace ptc::DirectX;

// using declarations
using ptc::Core::Error;
using ptc::Core::Format;




Primary::Primary(LPDIRECTDRAW ddraw,int pages,bool video)
{
    // setup ddraw
    m_ddraw = ddraw;

    // defaults
    m_width   = 0;
    m_height  = 0;
    m_back    = 0;
    m_front   = 0;
    m_palette = 0;
    m_surface = 0;
    
    try
    {
        // allocate surface array
        m_surface = new LPDIRECTDRAWSURFACE[pages];
        memset(m_surface,0,sizeof(LPDIRECTDRAWSURFACE)*pages);

        // create primary surface
        DDSURFACEDESC descriptor;
        descriptor.dwSize  = sizeof(descriptor);
        descriptor.dwFlags = DDSD_CAPS;
        descriptor.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
        if (video) descriptor.ddsCaps.dwCaps |= DDSCAPS_VIDEOMEMORY;
        check(m_ddraw->CreateSurface(&descriptor,&m_surface[0],0));

        // get surface descriptor
        check(m_surface[0]->GetSurfaceDesc(&descriptor));
        
        // get pixel format
        DDPIXELFORMAT ddpf;
        ddpf.dwSize = sizeof(ddpf);
        check(m_surface[0]->GetPixelFormat(&ddpf));

        // setup data
        m_front  = m_surface[0];
        m_pages  = pages;
        m_width  = descriptor.dwWidth;
        m_height = descriptor.dwHeight;
        m_format = translate(ddpf);

        // setup palette
        if (m_format.indexed())
        {
            // setup black palette
            PALETTEENTRY data[256];
            memset(data,0,256*sizeof(PALETTEENTRY));

            // create palette object
            check(m_ddraw->CreatePalette(DDPCAPS_8BIT|DDPCAPS_ALLOW256|DDPCAPS_INITIALIZE,data,&m_palette,0));

            // set palette of front surface
            check(m_front->SetPalette(m_palette));
        }

        // iterate number of pages
        for (int i=1; i<pages; i++)
        {
            // create offscreen surface
            descriptor.dwFlags  = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
            descriptor.dwWidth  = m_width;
            descriptor.dwHeight = m_height;
            if (!video) descriptor.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
            else descriptor.ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY;
            check(m_ddraw->CreateSurface(&descriptor,&m_surface[i],0));

            // attach to front surface
            check(m_front->AddAttachedSurface(m_surface[i]));
        }
        
        // check pages
        if (pages>1)
        {
            // get back surface
            DDSCAPS capabilities;
            capabilities.dwCaps = DDSCAPS_BACKBUFFER;
            check(m_front->GetAttachedSurface(&capabilities,&m_back));
        }
        else
        {
            // only one surface
            m_back = m_front;
        }
    }
    catch (Error &error)
    {
        // close
        close();

        // error message
        error.rethrow("could not create primary surface");
    }
}


Primary::~Primary()
{
    // close
    close();
}




void Primary::update()
{
    // flip surface
    if (m_front!=m_back) check(m_front->Flip(0,DDFLIP_WAIT));

    // sleep
    Sleep(0);
}




void Primary::write(void *pixels,int width,int height,const ptc::Base::Format &format,int palette[])
{
    // lock primary
    void *primary = lock();

    // create copy object
    Copy copy(format,m_format);

    // copy pixels to primary
    copy.copy(pixels,0,0,width,height,width*format.bits()/8,primary,0,0,m_width,m_height,pitch(),palette);

    // unlock
    unlock();
}




void* Primary::lock()
{
    // setup surface descriptor
    DDSURFACEDESC descriptor;
    descriptor.dwSize = sizeof descriptor;

    // lock surface
    check(m_back->Lock(0,&descriptor,DDLOCK_WAIT,0));

    // return locked buffer
    return descriptor.lpSurface;
}


void Primary::unlock()
{
    // unlock surface
    m_back->Unlock(0);
}




void Primary::palette(int32 palette[])
{
    // check format and palette
    if (!m_format.indexed() || !m_palette) return;

    // convert palette data
    PALETTEENTRY data[256];
    for (int i=0; i<256; i++)
    {
        data[i].peRed   = (palette[i] & 0x00FF0000) >> 16;
        data[i].peGreen = (palette[i] & 0x0000FF00) >> 8;
        data[i].peBlue  = (palette[i] & 0x000000FF) >> 0;
        data[i].peFlags = 0;
    }

    // set palette entries
    check(m_palette->SetEntries(0,0,256,data));
}




int Primary::width()
{
    // get width
    return m_width;
}


int Primary::height()
{
    // get height
    return m_height;
}


int Primary::pitch()
{
    // setup surface descriptor
    DDSURFACEDESC descriptor;
    descriptor.dwSize = sizeof(descriptor);

    // get surface descriptor
    check(m_back->GetSurfaceDesc(&descriptor));

    // return surface pitch
    return descriptor.lPitch;
}


const ptc::Base::Format& Primary::format()
{
    // get format
    return m_format;
}




Format Primary::translate(DDPIXELFORMAT const &ddpf)
{
    // pixel format
    Format format;

    // check rgb data
    if (ddpf.dwFlags & DDPF_PALETTEINDEXED8)
    {
        // indexed color
        format = Format(8);
    }
    else if (ddpf.dwFlags & DDPF_RGB)
    {
        // direct color
        format = Format(ddpf.dwRGBBitCount,ddpf.dwRBitMask,ddpf.dwGBitMask,ddpf.dwBBitMask);
    }
    else
    {
        // error message
        throw Error("invalid pixel format");
    }

    // return format
    return format;
}




void Primary::close()
{
    // wait for retrace
    if (m_ddraw) m_ddraw->WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN,0);

    // clear palette
    int32 data[256];
    memset(data,0,256*4);
    palette(data);

    // check surfaces
    if (m_front && m_back)
    {
        // clear
        DDBLTFX fx;
        fx.dwSize = sizeof(fx);
        fx.dwFillColor = 0;
        m_front->Blt(0,0,0,DDBLT_COLORFILL,&fx);
        update();
    }

    // iterate number of pages
    for (int i=0; i<m_pages; i++)
    {
        // release surface
        if (m_surface[i]) m_surface[i]->Release();
    }

    // delete array
    delete[] m_surface;
}
